<?php
/*
Plugin Name: Tidybot
Plugin URI: https://thevikingwebdesign.com
Description: A chatbot plugin for Tidycal with subscription handling.
Version: 1.1.76
Author: Daniel van der Velden
Author URI: https://thevikingwebdesign.com
License: GPL2
*/

if (!defined('ABSPATH')) exit;

define('TIDYBOT_ABSPATH', plugin_dir_path(__FILE__));
define('VIKING_API_SECRET_KEY', 'sk_x9xY0atkwO%gtMY#9%1yAtmWvB(uJuNU6qmEDtXuePSpYYkvB0(QVbv(EMDP#0eJ');
define('VIKING_API_PUBLIC_KEY', 'pk_1tEVKM4c2xGUkO0Si06DxaEUwMsfz1Es');

// Register cron schedule
add_filter('cron_schedules', 'add_fifteen_minute_interval');
function add_fifteen_minute_interval($schedules) {
    $schedules['fifteen_minutes'] = [
        'interval' => 900,
        'display' => __('Every 15 Minutes')
    ];
    return $schedules;
}

// Webhook check function
function tidybot_periodic_webhook_check() {
    $webhook_url = 'https://thevikingwebdesign.com/wp-content/check-subscription';
    $email = get_option('tidybot_email', '');

    if (empty($email)) {
        error_log('No email configured for webhook check');
        return;
    }

    $response = wp_remote_post($webhook_url, [
        'body' => json_encode(['email' => $email]),
        'headers' => [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . VIKING_API_SECRET_KEY,
            'X-API-Key' => VIKING_API_PUBLIC_KEY
        ],
        'timeout' => 30
    ]);

    if (is_wp_error($response)) {
        error_log('Webhook failed: ' . $response->get_error_message());
        return;
    }

    error_log('Webhook response: ' . wp_remote_retrieve_response_code($response) . ' - ' . wp_remote_retrieve_body($response));
}

// Schedule webhook check
add_action('init', 'schedule_tidybot_webhook_check');
function schedule_tidybot_webhook_check() {
    if (!wp_next_scheduled('tidybot_webhook_check')) {
        wp_schedule_event(time(), 'fifteen_minutes', 'tidybot_webhook_check');
    }
}

add_action('tidybot_webhook_check', 'tidybot_periodic_webhook_check');

// Cleanup on deactivation
register_deactivation_hook(__FILE__, 'clear_tidybot_webhook_schedule');
function clear_tidybot_webhook_schedule() {
    wp_clear_scheduled_hook('tidybot_webhook_check');
}

// Main subscription check function
function tidybot_check_subscription_via_webhook($email) {
    // Add debugging
    error_log('WEBHOOK CHECK: Checking subscription for email: ' . $email);
    
    $webhook_url = 'https://thevikingwebdesign.com/wp-content/check-subscription';

    $response = wp_remote_post($webhook_url, [
        'body' => json_encode(['email' => $email]),
        'headers' => [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . VIKING_API_SECRET_KEY,
            'X-API-Key' => VIKING_API_PUBLIC_KEY
        ],
        'timeout' => 30,
        'sslverify' => true
    ]);

    if (is_wp_error($response)) {
        error_log('WEBHOOK ERROR: ' . $response->get_error_message());
        return ['status' => 'error', 'message' => $response->get_error_message()];
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    
    error_log('WEBHOOK RESPONSE: Code=' . $response_code . ', Body=' . $body);

    $decoded_body = json_decode($body, true);
    if (!is_array($decoded_body) || !isset($decoded_body['status'])) {
        error_log('WEBHOOK ERROR: Invalid response format');
        return ['status' => 'error', 'message' => 'Invalid response format'];
    }

    // Log the final status
    error_log('WEBHOOK FINAL STATUS: ' . $decoded_body['status']);
    
    return $decoded_body;
}

// Asset loading
add_action('wp_enqueue_scripts', 'tidybot_enqueue_assets');
function tidybot_enqueue_assets() {
    if (!is_admin()) {
        wp_enqueue_style('tidybot-style', plugins_url('assets/tidybot.css', __FILE__));
        wp_enqueue_script('tidybot-script', plugins_url('assets/tidybot.js', __FILE__), ['jquery'], '1.0.1', true);
        wp_localize_script('tidybot-script', 'tidybotData', [
            'pluginUrl' => plugins_url('images/', __FILE__)
        ]);
    }
}

// Add chat widget
add_action('wp_footer', 'tidybot_add_dynamic_embed_script');
function tidybot_add_dynamic_embed_script() {
    $current_user = wp_get_current_user();
    
    // Check if user is logged in
    if (!$current_user->exists()) {
        error_log('TIDYBOT: No logged in user found');
        return;
    }
    
    // Gebruik de opgeslagen subscription status
    $subscription_status = get_option('tidybot_subscription_status', 'No subscription found');
    error_log('TIDYBOT: Using stored subscription status: ' . $subscription_status);

    global $wpdb;
    $table_name = $wpdb->prefix . 'tidycal_settings';
    $embed_code = $wpdb->get_var($wpdb->prepare("SELECT embed_code FROM $table_name WHERE id = %d", 1)) ?: '';
    
    error_log('TIDYBOT: Embed code found: ' . (!empty($embed_code) ? 'YES' : 'NO'));
    
    $position_class = get_option('tidybot_chat_position', 'bottom-right');
    $button_color = get_option('tidybot_button_color', '#006aa7');

    // Log the final decision
    error_log('TIDYBOT: Status=' . $subscription_status . ', Has embed code=' . (!empty($embed_code) ? 'yes' : 'no'));

    if ($subscription_status === 'active' && !empty($embed_code)) {
        error_log('TIDYBOT: Showing active chat');
        $output = sprintf(
            '<div id="tidybot-chat" class="%s" data-position="%s">%s</div>', 
            esc_attr($position_class),
            esc_attr($position_class),
            $embed_code
        );
        echo $output;
    } else {
        error_log('TIDYBOT: Showing inactive chat');
        printf(
            '<div id="tidybot-chat" class="%s" data-position="%s" style="background: %s; color: white; text-align: center; font-size: 18px; font-weight: bold; padding: 20px;">%s</div>',
            esc_attr($position_class),
            esc_attr($position_class),
            esc_attr($button_color),
            'Subscription Status: ' . esc_html($subscription_status)
        );
    }

    printf('<style>#tidybot-toggle{background-color:%s !important;}</style>', esc_attr($button_color));
}

// Admin menu
add_action('admin_menu', 'tidybot_add_admin_menu');
function tidybot_add_admin_menu() {
    add_menu_page(
        'Tidybot Settings',
        'Tidybot',
        'manage_options',
        'tidybot_settings',
        'tidybot_settings_page',
        'dashicons-admin-network'
    );
}

// Settings page
function tidybot_settings_page() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'tidycal_settings';

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['email'])) {
            $email = sanitize_email($_POST['email']);
            update_option('tidybot_email', $email);

            $subscription_data = tidybot_check_subscription_via_webhook($email);
            update_option('tidybot_subscription_status', $subscription_data['status']);
            update_option('tidybot_customer_id', $subscription_data['customer_id'] ?? '');
        }

        if (isset($_POST['embed_code'])) {
            $embed_code = stripslashes($_POST['embed_code']);
            $exists = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE id = %d", 1));

            if ($exists) {
                $wpdb->update($table_name, ['embed_code' => $embed_code], ['id' => 1], ['%s'], ['%d']);
            } else {
                $wpdb->insert($table_name, ['id' => 1, 'embed_code' => $embed_code], ['%d', '%s']);
            }
        }

        if (isset($_POST['chat_position'])) {
            update_option('tidybot_chat_position', sanitize_text_field($_POST['chat_position']));
        }

        if (isset($_POST['button_color'])) {
            $button_color = sanitize_hex_color($_POST['button_color']);
            if ($button_color) {
                update_option('tidybot_button_color', $button_color);
            }
        }
    }

    $email = get_option('tidybot_email', '');
    $subscription_status = get_option('tidybot_subscription_status', 'No subscription found');
    $customer_id = get_option('tidybot_customer_id', '');
    $embed_code = $wpdb->get_var($wpdb->prepare("SELECT embed_code FROM $table_name WHERE id = %d", 1)) ?: '';
    $button_color = get_option('tidybot_button_color', '#006aa7');
    $chat_position = get_option('tidybot_chat_position', 'bottom-right');

    ?>
    <div class="wrap">
        <h1>Tidybot Settings</h1>
        <form method="POST">
            <h2>Email</h2>
            <input type="email" name="email" value="<?php echo esc_attr($email); ?>" required style="width: 300px;">

            <h2>Subscription Status</h2>
            <p style="color: <?php echo ($subscription_status === 'active') ? 'green' : 'red'; ?>; font-weight: bold;">
                <?php echo esc_html($subscription_status); ?>
            </p>

            <?php if ($customer_id) : ?>
                <h2>Customer ID</h2>
                <p><?php echo esc_html($customer_id); ?></p>
            <?php endif; ?>

            <h2>Embed Code</h2>
            <textarea name="embed_code" rows="5" cols="50"><?php echo esc_textarea($embed_code); ?></textarea>

            <h2>Button Color</h2>
            <input type="text" name="button_color" value="<?php echo esc_attr($button_color); ?>" placeholder="#123456">

            <h2>Chat Position</h2>
            <select name="chat_position">
                <option value="bottom-right" <?php selected($chat_position, 'bottom-right'); ?>>Bottom Right</option>
                <option value="bottom-left" <?php selected($chat_position, 'bottom-left'); ?>>Bottom Left</option>
            </select>

            <p><button type="submit" class="button-primary">Save Settings</button></p>
        </form>
    </div>
    <?php
}

// Database setup
register_activation_hook(__FILE__, 'tidybot_create_table');
function tidybot_create_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'tidycal_settings';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id INT(11) NOT NULL AUTO_INCREMENT,
        embed_code TEXT NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    if (!$wpdb->get_var("SELECT COUNT(*) FROM $table_name")) {
        $wpdb->insert($table_name, ['id' => 1, 'embed_code' => ''], ['%d', '%s']);
    }
}

// Voeg deze functie toe om de embed code te controleren
add_action('admin_init', 'check_tidybot_embed_code');
function check_tidybot_embed_code() {
    if (current_user_can('manage_options')) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'tidycal_settings';
        $embed_code = $wpdb->get_var($wpdb->prepare("SELECT embed_code FROM $table_name WHERE id = %d", 1));
        error_log('Admin Check - Embed code in database: ' . $embed_code);
    }
}