<?php
// Security check
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generate API keys for Viking integrations
 */
function theviking_generate_api_keys($force_regenerate = false) {
    // Check if the API keys already exist
    $public_key = get_option('tidybot_api_public_key');
    $secret_key = get_option('tidybot_api_secret_key');

    if (!$public_key || !$secret_key || $force_regenerate) {
        // Generate new API keys
        $public_key = 'pk_' . wp_generate_password(32, false); // Public key
        $secret_key = 'sk_' . wp_generate_password(64, true);  // Secret key

        // Save in WordPress options
        update_option('tidybot_api_public_key', $public_key);
        update_option('tidybot_api_secret_key', $secret_key);
    }

    return [
        'public_key' => $public_key,
        'secret_key' => $secret_key,
    ];
}

/**
 * Display API keys in the admin dashboard
 */
function theviking_api_keys_page() {
    // Regenerate keys if requested
    if (isset($_POST['regenerate_keys'])) {
        if (!check_admin_referer('theviking_regenerate_keys_action', 'theviking_nonce')) {
            wp_die('Invalid request.');
        }

        theviking_generate_api_keys(true);
        echo '<div class="updated"><p>API keys regenerated successfully!</p></div>';
    }

    // Retrieve existing API keys
    $keys = theviking_generate_api_keys();
    $masked_secret_key = substr($keys['secret_key'], 0, 4) . str_repeat('*', strlen($keys['secret_key']) - 8) . substr($keys['secret_key'], -4);

    ?>
    <div class="wrap">
        <h1>API Settings</h1>
        <p>Here are your API keys. Keep the Secret Key safe and do not share it with others.</p>
        <table class="form-table">
            <tr>
                <th>Public Key:</th>
                <td>
                    <code class="viking-api-key" onclick="copyToClipboard(this)"><?php echo esc_html($keys['public_key']); ?></code>
                </td>
            </tr>
            <tr>
                <th>Secret Key:</th>
                <td>
                    <code class="viking-api-key"><?php echo esc_html($masked_secret_key); ?></code>
                    <p><em>The full key will only be displayed when regenerated.</em></p>
                </td>
            </tr>
        </table>
        <form method="post">
            <?php wp_nonce_field('theviking_regenerate_keys_action', 'theviking_nonce'); ?>
            <button type="submit" name="regenerate_keys" class="button button-primary">Regenerate Keys</button>
        </form>
    </div>
    <script>
        function copyToClipboard(element) {
            const tempInput = document.createElement('input');
            tempInput.value = element.textContent;
            document.body.appendChild(tempInput);
            tempInput.select();
            document.execCommand('copy');
            document.body.removeChild(tempInput);
            alert('Key copied to clipboard!');
        }
    </script>
    <style>
        .viking-api-key {
            display: inline-block;
            padding: 4px 8px;
            background-color: #f4f4f4;
            border: 1px solid #ddd;
            border-radius: 4px;
            cursor: pointer;
        }
    </style>
    <?php
}
